/*-------------------------------------------------------*
*  Name:      VRL filter for ZModeler v1.05              *
*  Purpose:   SCGT & Formula1 2000 files import/export   *
*  Authors:   Oleg M.                                    *
*                                                        *
*  History:   16.Sep.2001 - rebuild for ZModeler v1.05   *
*  History:   12.May.2002 - Multiple files export added  *
*                                                        *
*-------------------------------------------------------*/
#include <Struct.h>
#include <3DEngine.h>
#include <ZMUserAddons/ZMultipleFilesExport.h>

#include "resource.h"

#include "VRL.h"
#include "VRLType.h"


//////////////////////////////////////////////////////////////
//  DWORD Capabilities - a combination versions
DWORD CALLBACK Capabilities(long)
{
  return ZMODELER_BUILD_VERSION();
}

//////////////////////////////////////////////////////////////
//  DWORD Supports returns a combination of supported features
DWORD CALLBACK Supports(DWORD)
{
  return Z3D_PLUGSUPPORTS_IMPORT | Z3D_PLUGSUPPORTS_EXPORT |
         Z3D_PLUGSUPPORTS_MISCFLAGS;
}

//////////////////////////////////////////////////////////////
//  DWORD DynamicLoading specifies whether the dll will be
//  loaded only when it'sfunctions are needed.
//  You should NOT make it dynamic, if it is a processor-type.
DWORD CALLBACK DynamicLoading(DWORD)
{
  return 1L; // DLL will be loaded each time it is needed
}

//////////////////////////////////////////////////////////////
//  void GetFilterExtension returns a file extention for sup-
//  ported files. Used if it is an IMPORT/EXPORT type
char* CALLBACK GetFilterExtension(DWORD)
{
  return "vrl";
}

//////////////////////////////////////////////////////////////
//  void GetFilterMask returns a file-mask for CFileDilaog.
//  Used if it is an IMPORT/EXPORT type.
char* CALLBACK GetFilterMask(DWORD)
{
  return "SCGT & Formula-1 2000 (*.vrl)|*.vrl|";
}



//////////////////////////////////////////////////////////////
//  Shows a failure message.
DWORD _stdcall ShowFailMessage(CWnd* pwnd, char *errmsg, char *cause, UINT icon)
{
  char str[512];
  strcpy(str, "Error importing/exporting file: ");
  strcat(str, errmsg);
  strcat(str, "\nPossible caused by: ");
  strcat(str, cause);
  pwnd->MessageBox(str,"Filter message:", icon);
  return 0;
}



char* CALLBACK MiscellaneousFlagsMeaning(DWORD num)
{
  if (num!= 0) return NULL;
  return
    "SCGT faces properties VRL filter\n"
    "\n\n\n\n"
    "Enable Transparency through RGBA\n\n\n\n"
    "\n\n\n\n";
}


void GetPlaneCoefficients(VRLVertex P1,
              VRLVertex P2,
              VRLVertex P3,
              float *A,
              float *B,
              float *C)
{
  float AX, AY, AZ, BX, BY, BZ, CX, CY, CZ;
  AX = P1.X+P2.X;
  BX = P2.X+P3.X;
  CX = P3.X+P1.X;
  AY = P1.Y+P2.Y;
  BY = P2.Y+P3.Y;
  CY = P3.Y+P1.Y;
  AZ = P1.Z+P2.Z;
  BZ = P2.Z+P3.Z;
  CZ = P3.Z+P1.Z;
  *A = (P1.Y-P2.Y)*AZ+(P2.Y-P3.Y)*BZ+(P3.Y-P1.Y)*CZ;
  *B = (P1.Z-P2.Z)*AX+(P2.Z-P3.Z)*BX+(P3.Z-P1.Z)*CX;
  *C = (P1.X-P2.X)*AY+(P2.X-P3.X)*BY+(P3.X-P1.X)*CY;
}


float GetDPlane(VRLVertex P1, VRLVertex P2, VRLVertex P3)
{
    float A,B,C,DPlane, UL;

  GetPlaneCoefficients(P1,P2,P3, &A, &B, &C);
  UL = (float)sqrt(A*A+B*B+C*C);
    DPlane = -(A*P1.X+B*P1.Y+C*P1.Z);
    if (UL != 0) DPlane = DPlane/UL;
    else         DPlane = 0.0f;
    
  return DPlane;
}



/**********************************************************************/
//    SCGT & Formula1 2000 File Importer to Zanoza 3D Editor
/**********************************************************************/
DWORD CALLBACK Import(  CString fromfile,
            CWnd *pwnd,
            tObjectSet* Objects,
            tUnDataSet* UnData,
            CDirect3D*  d3d,
            SYSTEMREQUESTPROC RequestProc,
            HINSTANCE AppHIns,
            HINSTANCE DllHIns)
{
  CFile InFile;
  if (!InFile.Open(fromfile, CFile::modeRead))
    return ShowFailMessage(pwnd, "can not open the specified file.", "invalid file attributes or share violation.", MB_ICONHAND);
  

  //Declaring all neccessary variables and arrays;
  CString    MatNameStr, TextureStr, TempStr;
  VRLHeader  Header;
  fVRLHeader  fHeader;
  VRLVertex  *verts, *norms;
  long i, j, numOf;
  long    *FirstNorms,
        *NormCounts,
        *NormIndexes,
        *FirstPolys,
        *PolyCounts,
        *TexMapMaterial,
        *TexIndexes,
        *PolyTypes;
  VRLPoint  VPBlock;
  VRLColor  *ColorMap;
  TextureName *Textures;

  long NumVerts, NumNorms, NumPoints, NumPoly, NumTex;

  long type;

  AfxSetResourceHandle(DllHIns);
  CVRLTypeDialog diag(&type);
  if (diag.DoModal() != IDOK)
  {
    AfxSetResourceHandle(AppHIns);
    return 0;
  }
  AfxSetResourceHandle(AppHIns);

  tObject* obj;
  obj = new tObject;

  if (type == 1)
  {
    InFile.Read(&Header, sizeof(VRLHeader));
    NumVerts = Header.VertexHeader.VertexCount;
    NumNorms = Header.NormalHeader.NormalCount;
    NumPoints= Header.PointHeader.PointCount;
    NumPoly  = Header.PolygonHeader.PolygonCount;
    NumTex   = Header.TextureMapHeader.TextureMapCount;
  }
  else
  {
    InFile.Read(&fHeader, sizeof(fVRLHeader));
    NumVerts = fHeader.VertexHeader.VertexCount;
    NumNorms = fHeader.NormalHeader.NormalCount;
    NumPoints= fHeader.PointHeader.PointCount;
    NumPoly  = fHeader.PolygonHeader.PolygonCount;
    NumTex   = fHeader.TextureMapHeader.TextureMapCount;
  }

  // Creating the vertices block in new Object
  obj->VertTable->VertAmount = NumVerts;
  obj->VertTable->Table = new tNormalVertex[NumVerts];

  // creating arrays for reading from files
  verts = new VRLVertex[NumVerts];
  norms = new VRLVertex[NumNorms];
  
  if (type == 1)
  {
    NormCounts = new long[NumVerts];
    FirstNorms = new long[NumVerts];
  }


  FirstPolys  = new long[NumPoly];
  PolyCounts  = new long[NumPoly];
  PolyTypes  = new long[NumPoly];
  ColorMap  = new VRLColor[NumPoly];
  TexIndexes  = new long[NumPoly];

  VPBlock.VertIndexes  = new long[NumPoints];
  VPBlock.Us      = new float[NumPoints];
  VPBlock.Vs      = new float[NumPoints];
  
  Textures        = new TextureName[NumTex];
  TexMapMaterial  = new long[NumTex+1]; TexMapMaterial[NumTex] = 0;


  if (type == 1)
  {
    //Read: Vertices: X,Y,Z
    InFile.Seek(Header.VertexHeader.VertexsOffset, CFile::begin);
    InFile.Read(verts, sizeof(VRLVertex)*NumVerts);
    //Read: Vertices: FirstNormals
    InFile.Seek(Header.VertexHeader.FirstGNormalsOffset, CFile::begin);
    InFile.Read(FirstNorms, 4*NumVerts);
    //Read: Vertices: NormalsCounts
    InFile.Seek(Header.VertexHeader.GNormalIndexCountOffset, CFile::begin);
    InFile.Read(NormCounts, 4*NumVerts);
  }
  else
  {
    //Read: Vertices: X,Y,Z
    InFile.Seek(fHeader.VertexHeader.VertexsOffset, CFile::begin);
    InFile.Read(verts, sizeof(VRLVertex)*NumVerts);
  }

  //Read: Points: VertexIndexes
  if (type == 1)  InFile.Seek(Header.PointHeader.PointsOffset, CFile::begin);
  else      InFile.Seek(fHeader.PointHeader.PointsOffset, CFile::begin);
  InFile.Read(VPBlock.VertIndexes, 4*NumPoints);

  //Read: Points: Us
  if (type == 1)  InFile.Seek(Header.PointHeader.TUPointsOffset, CFile::begin);
  else      InFile.Seek(fHeader.PointHeader.TUPointsOffset, CFile::begin);
  InFile.Read(VPBlock.Us, 4*NumPoints);

  //Read: Points: Us
  if (type == 1)  InFile.Seek(Header.PointHeader.TVPointsOffset, CFile::begin);
  else      InFile.Seek(fHeader.PointHeader.TVPointsOffset, CFile::begin);
  InFile.Read(VPBlock.Vs, 4*NumPoints);

  //Read: Polygons: FirstPolys
  if (type == 1)  InFile.Seek(Header.PolygonHeader.FirstPointsOffset, CFile::begin);
  else      InFile.Seek(fHeader.PolygonHeader.FirstPointsOffset, CFile::begin);
  InFile.Read(FirstPolys, 4*NumPoly);

  //Read: Polygons: PolyCounts
  if (type == 1)  InFile.Seek(Header.PolygonHeader.PointCountsOffset, CFile::begin);
  else      InFile.Seek(fHeader.PolygonHeader.PointCountsOffset, CFile::begin);
  InFile.Read(PolyCounts, 4*NumPoly);

  //Read: Polygons: Types: textured/not.
  if (type == 1)  InFile.Seek(Header.PolygonHeader.ColorsOffset, CFile::begin);
  else      InFile.Seek(fHeader.PolygonHeader.ColorsOffset, CFile::begin);
  InFile.Read(ColorMap, sizeof(VRLColor)*NumPoly);

  //Read: Polygons: Types: textured/not.
  if (type == 1)  InFile.Seek(Header.PolygonHeader.TypesOffset, CFile::begin);
  else      InFile.Seek(fHeader.PolygonHeader.TypesOffset, CFile::begin);
  InFile.Read(PolyTypes, 4*NumPoly);

  //Read: Polygons: TextureFiles
  if (type == 1)  InFile.Seek(Header.PolygonHeader.TexIndexesOffset, CFile::begin);
  else      InFile.Seek(fHeader.PolygonHeader.TexIndexesOffset, CFile::begin);
  InFile.Read(TexIndexes, 4*NumPoly);

  //Read: Textures: filenames
  if (type == 1)  InFile.Seek(Header.TextureMapHeader.TextureMapsOffset, CFile::begin);
  else      InFile.Seek(fHeader.TextureMapHeader.TextureMapsOffset, CFile::begin);
  InFile.Read(Textures, sizeof(TextureName)*NumTex);


  for (i = 0; i < NumTex; i++)
  {//for
    TextureStr = Textures[i];
    if (TextureStr.Find(' ') > 0)
      TextureStr = TextureStr.Left(TextureStr.Find(' '));
    TextureStr += ".bmp";
    MatNameStr = "Textured Material: ";
    MatNameStr += TextureStr;
    for (j = 0; j < d3d->Materials.MaterialsAmount; j++)
    {//checkin whether this material is already presant
      TempStr = d3d->Materials.Materials[j].MaterialName;
      if (TempStr.Find(TextureStr) >= 0)
      {// the texture is present
        TexMapMaterial[i] = j;
        j = d3d->Materials.MaterialsAmount+1;
      }// the texture is present
    }//checkin whether this material is already presant
    if (j <= d3d->Materials.MaterialsAmount)
      TexMapMaterial[i] = d3d->CreateMaterial(
            1.0f,//float  red,
            1.0f,//float  green,
            1.0f,//float  blue,
            1.0f,//alpha
            1.0f,//float  shine,
            80.0f,//float  shinestrength,
            (char*)TextureStr.GetBuffer(MAX_MATERIALNAMELEN),
            "",
            "chrome.bmp",
            "",
            (char*)MatNameStr.GetBuffer(MAX_MATERIALNAMELEN),
            0,//TEX_CONFIRMRELOAD,
            D3DTOP_MODULATE ,
            D3DTOP_DISABLE,
            D3DTOP_ADD,
            D3DTOP_DISABLE,
            D3DBLEND_SRCALPHA,
            D3DBLEND_INVSRCALPHA,
            0,  //unused
            0x00,
            1);
  }//for
  //remapping texture indexes to material indexes
  for (i = 0; i < NumPoly; i++)
    TexIndexes[i] = TexMapMaterial[TexIndexes[i]];
  //remapping color indexes to material indexes
  for (i = 0; i < NumPoly; i++)
    ColorMap[i].Alpha = 0;


  for (i = 0; i < NumPoly; i++)
    if (ColorMap[i].Alpha == 0)
    {//add this color to the materials set
      for (j = 0; j < d3d->Materials.MaterialsAmount; j++)
      {//checkin whether this RGB material is already presant
        if (d3d->Materials.Materials[j].MatParams.PrimTexture==-1)
        {//RGB
          if ((d3d->Materials.Materials[j].MatRec.dcvAmbient.r == (float)ColorMap[i].R/255.0f) &&
            (d3d->Materials.Materials[j].MatRec.dcvAmbient.g == (float)ColorMap[i].G/255.0f) &&
            (d3d->Materials.Materials[j].MatRec.dcvAmbient.b == (float)ColorMap[i].B/255.0f))
          {
            ColorMap[i].Alpha = (BYTE)j;
            j = d3d->Materials.MaterialsAmount+1;
          }
        }//RGB
      }//checkin whether this RGB material is already presant
      if (j <= d3d->Materials.MaterialsAmount)
        ColorMap[i].Alpha = (BYTE)d3d->CreateMaterial(
                (float)ColorMap[i].R/255.0f,
                (float)ColorMap[i].G/255.0f,
                (float)ColorMap[i].B/255.0f,
                1.0f,//alpha
                1.0f,//float shine,
                80.0f,//float shinestrength,
                "",//NO TEXTURE
                "",
                "",
                "",
                "Colored Material",
                0x00000000,
                D3DTOP_DISABLE,
                D3DTOP_DISABLE,
                D3DTOP_DISABLE,
                D3DTOP_DISABLE,
                D3DBLEND_SRCALPHA,
                D3DBLEND_INVSRCALPHA,
                0, //usused
                0x00,
                1);
      for (j = i + 1; j < NumPoly; j++)
        if ((ColorMap[j].Alpha==0) && (ColorMap[j].R==ColorMap[i].R)&& (ColorMap[j].G==ColorMap[i].G)&& (ColorMap[j].B==ColorMap[i].B))
          ColorMap[j].Alpha = ColorMap[i].Alpha;
    }//add this color to the materials set
  //end mapping matte colors

  numOf = 0;
  if (type == 1)
    for (i = 0; i < NumVerts; i++)
      numOf += NormCounts[i];
  
  if (type == 2) numOf = NumPoly;
  NormIndexes = new long[numOf];

  if (type == 1)
  {
    //Read: Vertices: NormalsIndexes
    InFile.Seek(Header.VertexHeader.GNormalIndexesOffset, CFile::begin);
    InFile.Read(NormIndexes, 4*numOf);
  }
  else
  {
    //Read: Vertices: NormalsIndexes
    InFile.Seek(fHeader.PolygonHeader.NormalIndexesOffset, CFile::begin);
    InFile.Read(NormIndexes, 4*NumPoly);
  }

  //Read: Normals: XYZ
  if (type == 1)  InFile.Seek(Header.NormalHeader.NormalsOffset, CFile::begin);
  else      InFile.Seek(fHeader.NormalHeader.NormalsOffset, CFile::begin);
  InFile.Read(norms, sizeof(VRLVertex)*NumNorms);

    
    
    
  //Vertices: XYZ
  for (i = 0; i < NumVerts; i++)
  {
    obj->VertTable->Table[i].X = verts[i].X;
    obj->VertTable->Table[i].Y = verts[i].Y;
    obj->VertTable->Table[i].Z = -verts[i].Z;
    obj->VertTable->Table[i].NormalX = 0.0f;
    obj->VertTable->Table[i].NormalY = 0.0f;
    obj->VertTable->Table[i].NormalZ = 0.0f;
  }
  //Normals: XYZ
  if (type==1)
    for (i = 0; i < NumVerts; i++)
    {
      for (j = 0; j < NormCounts[i]; j++)
      {
        obj->VertTable->Table[i].NormalX += norms[NormIndexes[FirstNorms[i]+j]].X;
        obj->VertTable->Table[i].NormalY += norms[NormIndexes[FirstNorms[i]+j]].Y;
        obj->VertTable->Table[i].NormalZ += norms[NormIndexes[FirstNorms[i]+j]].Z;
      }
      obj->VertTable->Table[i].NormalX = -obj->VertTable->Table[i].NormalX/(float)NormCounts[i];
      obj->VertTable->Table[i].NormalY = -obj->VertTable->Table[i].NormalY/(float)NormCounts[i];
      obj->VertTable->Table[i].NormalZ = obj->VertTable->Table[i].NormalZ/(float)NormCounts[i];
    }

  //counting an amount of faces
  numOf = 0;
  for (i = 0; i < NumPoly; i++)
    numOf += (PolyCounts[i]-2);

  //Creating Faces
  obj->FaceTable->FaceAmount = numOf;
  obj->FaceTable->Table = new tFace[numOf];
  //Creating the buffer for material indexes;
  
  numOf = 0;
  for (i = 0; i < NumPoly; i++)
  {
    //FirstPolys;
    //PolyCounts;
    obj->FaceTable->Table[i+numOf].I1 = VPBlock.VertIndexes[FirstPolys[i]];
    obj->FaceTable->Table[i+numOf].I2 = VPBlock.VertIndexes[FirstPolys[i]+2];
    obj->FaceTable->Table[i+numOf].I3 = VPBlock.VertIndexes[FirstPolys[i]+1];
    obj->FaceTable->Table[i+numOf].U1 = (float)fabs(VPBlock.Us[FirstPolys[i]]);
    obj->FaceTable->Table[i+numOf].U2 = (float)fabs(VPBlock.Us[FirstPolys[i]+2]);
    obj->FaceTable->Table[i+numOf].U3 = (float)fabs(VPBlock.Us[FirstPolys[i]+1]);
    obj->FaceTable->Table[i+numOf].V1 = (float)fabs(VPBlock.Vs[FirstPolys[i]]);
    obj->FaceTable->Table[i+numOf].V2 = (float)fabs(VPBlock.Vs[FirstPolys[i]+2]);
    obj->FaceTable->Table[i+numOf].V3 = (float)fabs(VPBlock.Vs[FirstPolys[i]+1]);


    if (type == 1)
    {//scgt types
      if (PolyTypes[i] == 0x24)
        obj->FaceTable->Table[i+numOf].Material = ColorMap[i].Alpha;//untextured
      else        
        obj->FaceTable->Table[i+numOf].Material = TexIndexes[i];
      if (PolyTypes[i] & 0x0800)
        obj->FaceTable->Table[i+numOf].nRenderFlags = SemiTransparent;
      else
        obj->FaceTable->Table[i+numOf].nRenderFlags = 0L;
      if ((PolyTypes[i] & 0x02) == 0x02)//with transparency tyrough rgbA
        obj->FaceTable->Table[i+numOf].SetFlag(Z3D_FLAG_MISC1);
    }//scgt types
    else
    {//f2000 types
      obj->FaceTable->Table[i+numOf].Material = TexIndexes[i];
      if ((PolyTypes[i] & 0x9000)!=0x9000)//no blend
        obj->FaceTable->Table[i+numOf].nRenderFlags = NoBlend;
      if (PolyTypes[i] & 0x0800)//semitransparent
        obj->FaceTable->Table[i+numOf].nRenderFlags |= SemiTransparent;
    }//f2000 types
    
    if (type == 2)    
    {//update normals;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalX = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalX -  norms[NormIndexes[i]].X)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalY = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalY -  norms[NormIndexes[i]].Y)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalZ = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalZ +  norms[NormIndexes[i]].Z)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalX = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalX -  norms[NormIndexes[i]].X)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalY = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalY -  norms[NormIndexes[i]].Y)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalZ = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalZ +  norms[NormIndexes[i]].Z)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalX = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalX -  norms[NormIndexes[i]].X)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalY = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalY -  norms[NormIndexes[i]].Y)/2.0f;
      obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalZ = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalZ +  norms[NormIndexes[i]].Z)/2.0f;
    }
    for (j = 3; j < PolyCounts[i]; j++)
    {//non 3 vert poly
      numOf++;
      obj->FaceTable->Table[i+numOf].I1 = VPBlock.VertIndexes[FirstPolys[i]];
      obj->FaceTable->Table[i+numOf].I2 = VPBlock.VertIndexes[FirstPolys[i]+j];
      obj->FaceTable->Table[i+numOf].I3 = VPBlock.VertIndexes[FirstPolys[i]+j-1];
      obj->FaceTable->Table[i+numOf].U1 = (float)fabs(VPBlock.Us[FirstPolys[i]]);
      obj->FaceTable->Table[i+numOf].U2 = (float)fabs(VPBlock.Us[FirstPolys[i]+j]);
      obj->FaceTable->Table[i+numOf].U3 = (float)fabs(VPBlock.Us[FirstPolys[i]+j-1]);
      obj->FaceTable->Table[i+numOf].V1 = (float)fabs(VPBlock.Vs[FirstPolys[i]]);
      obj->FaceTable->Table[i+numOf].V2 = (float)fabs(VPBlock.Vs[FirstPolys[i]+j]);
      obj->FaceTable->Table[i+numOf].V3 = (float)fabs(VPBlock.Vs[FirstPolys[i]+j-1]);
      if (type == 1)
      {//scgt types
        if (PolyTypes[i] == 0x24)  obj->FaceTable->Table[i+numOf].Material = ColorMap[i].Alpha;
        else            obj->FaceTable->Table[i+numOf].Material = TexIndexes[i];
        if (PolyTypes[i] & 0x0800)
          obj->FaceTable->Table[i+numOf].nRenderFlags = SemiTransparent;
        else
          obj->FaceTable->Table[i+numOf].nRenderFlags = 0L;
        if ((PolyTypes[i] & 0x02) == 0x02)//with transparency tyrough rgbA
          obj->FaceTable->Table[i+numOf].SetFlag(Z3D_FLAG_MISC1);
      }//scgt types
      else
      {//f2000 types
        obj->FaceTable->Table[i+numOf].Material = TexIndexes[i];
        if ((PolyTypes[i] & 0x9000)!=0x9000)//no blend
          obj->FaceTable->Table[i+numOf].nRenderFlags = NoBlend;
        if (PolyTypes[i] & 0x0800)//semitransparent
          obj->FaceTable->Table[i+numOf].nRenderFlags |= SemiTransparent;
      }//f2000 types
      if (type == 2)
      {
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalX = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalX -  norms[NormIndexes[i]].X)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalY = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalY -  norms[NormIndexes[i]].Y)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalZ = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I1].NormalZ +  norms[NormIndexes[i]].Z)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalX = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalX -  norms[NormIndexes[i]].X)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalY = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalY -  norms[NormIndexes[i]].Y)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalZ = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I2].NormalZ +  norms[NormIndexes[i]].Z)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalX = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalX -  norms[NormIndexes[i]].X)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalY = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalY -  norms[NormIndexes[i]].Y)/2.0f;
        obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalZ = (obj->VertTable->Table[obj->FaceTable->Table[i+numOf].I3].NormalZ +  norms[NormIndexes[i]].Z)/2.0f;
      }
    }//non 3 vert poly
  }


  if (fromfile.ReverseFind('\\') > 0)
    fromfile = fromfile.Right(fromfile.GetLength() - fromfile.ReverseFind('\\')-1);

  strcpy(obj->ObjectName, fromfile);
  
  //normalising
  obj->VertTable->MarkAll();
  obj->NormalizeNormals(TRUE, FALSE);
  obj->VertTable->UnMarkAll();

  obj->UpdateBoundary();
  Objects->AddObject(obj);
  delete obj;
  ///////////////////// Finish
  InFile.Close();
  delete[] verts;
  delete[] norms;
  if (type == 1)
  {
    delete[] NormCounts;
    delete[] FirstNorms;
  }
  delete[] NormIndexes;
  delete[] VPBlock.VertIndexes;
  delete[] VPBlock.Us;
  delete[] VPBlock.Vs;
  delete[] FirstPolys;
  delete[] PolyCounts;
  delete[] PolyTypes;
  delete[] ColorMap;
  delete[] TexIndexes;
  delete[] Textures;
  delete[] TexMapMaterial;

  return 1;
}//Import




/**********************************************************************/
//      SCGT File Exporter to Zanoza 3D Editor
/**********************************************************************/
DWORD CALLBACK MultipleExport(CFile &OutFile, tObject* pobj, CDirect3D* pD3D, long type);

DWORD CALLBACK Export(  CString Tofile,
            CWnd* pwnd,
            tObjectSet* Objects,
            tUnDataSet* UnData,
            CDirect3D*  d3d,
            SYSTEMREQUESTPROC RequestProc,
            HINSTANCE AppHIns,
            HINSTANCE DllHIns)
{
  CFile OutFile;
  if (!OutFile.Open(Tofile, CFile::modeCreate | CFile::modeWrite))
    return ShowFailMessage(pwnd, "can not open the specified file.", "invalid file attributes or share violation.", MB_ICONHAND);

  //Now I'll show a dialogBox, where an object for exporting
  //will be selected by user.
  if (!RequestProc)
    return ShowFailMessage(pwnd, "Invalid parameters passed to plugin.", "Versions mismatch.", MB_ICONHAND);
  
  if ((WORD)(SYSTEMREQUESTPROC)RequestProc(
          Z3D_REQUEST_SELECT_BYNAME,
          Objects,
          "Select object for exporting",
          NULL,
          NULL)==IDCANCEL)
    return 0;

  // Now we are ready to export the selected file.
  long type;

  AfxSetResourceHandle(DllHIns);
  CVRLTypeDialog diag(&type);
  if (diag.DoModal() != IDOK)
  {
    AfxSetResourceHandle(AppHIns);
    return 0;
  }
  AfxSetResourceHandle(AppHIns);

  for (long count=0; count < Objects->ObjAmount; count++)
    if (Objects->ObjSet[count].Selected())
    {
      MultipleExport(OutFile, &Objects->ObjSet[count], d3d, type);
      break;
    }
  OutFile.Close();
  return 0;
}//Export



DWORD CALLBACK MultipleExport(CFile &OutFile, tObject* pobj, CDirect3D* d3d, long type)
{
  // All necessary variables are here;
  VRLHeader   Header;    //scgt
  fVRLHeader  fHeader;  //f1 2000
  VRLVertex  *verts, *norms;
  long  *VertZERObuf,
        *VertIntens,
        *FirstNorms,
        *NormCounts,
        *NormIndexes,
        *PolyCounts,
        *FirstPolys,
        *NormPolys,
        *FlagsPolys,
        *ColorsPolys,
        *MIPPolys,
        *TypesPolys,
        *TexIndexes;
  float    *DPlanePolys;
  float    *PointsTS;
  long    *PolyZeros;

  VRLPoint  VPBlock;
  long    i, j, index;
  long    MaterialMap[255];
  TextureName*TextureNames = NULL;
  CString    str;
  tPOINT    Normal;
  long NumVerts, NumPoly, NumPoints, NumNorms, NumTex;
  
  if (type == 1)
  {
    for (j = 0; j < 60; j++)
      Header.m_Blank[j] = '\0';
    strcpy((char*)&Header.m_Blank[0], "ZModeler generated file");
  }

  if (type == 1)
  {
    Header.VertexHeader.VertexCount        = pobj->VertTable->VertAmount;
    Header.VertexHeader.VertexsOffset      = 0xAC;
    Header.VertexHeader.TransformedVerexOffset  = Header.VertexHeader.VertexsOffset + Header.VertexHeader.VertexCount*sizeof(VRLVertex);
    Header.VertexHeader.VertexIntensitysOffset  = Header.VertexHeader.TransformedVerexOffset + Header.VertexHeader.VertexCount*32;
    Header.VertexHeader.VertexAIntensitysOffset  = Header.VertexHeader.VertexIntensitysOffset + Header.VertexHeader.VertexCount*4;
    Header.VertexHeader.VertexBIntensitysOffset  = Header.VertexHeader.VertexAIntensitysOffset + Header.VertexHeader.VertexCount*4;
    Header.VertexHeader.FirstGNormalsOffset    = Header.VertexHeader.VertexBIntensitysOffset + Header.VertexHeader.VertexCount*4;
    Header.VertexHeader.GNormalIndexCountOffset = Header.VertexHeader.FirstGNormalsOffset + Header.VertexHeader.VertexCount*4;
    Header.VertexHeader.GNormalIndexesOffset  = Header.VertexHeader.GNormalIndexCountOffset + Header.VertexHeader.VertexCount*4;

    Header.PointHeader.PointCount        = pobj->FaceTable->FaceAmount*3;
    Header.PointHeader.PointsOffset        = Header.VertexHeader.GNormalIndexesOffset + Header.VertexHeader.VertexCount*4;
    Header.PointHeader.TUPointsOffset      = Header.PointHeader.PointsOffset + Header.PointHeader.PointCount*4;
    Header.PointHeader.TVPointsOffset      = Header.PointHeader.TUPointsOffset + Header.PointHeader.PointCount*4;

    Header.NormalHeader.NormalCount        = pobj->VertTable->VertAmount + pobj->FaceTable->FaceAmount;
    Header.NormalHeader.NormalsOffset      = Header.PointHeader.TVPointsOffset + Header.PointHeader.PointCount*4;

    Header.PolygonHeader.PolygonCount      = pobj->FaceTable->FaceAmount;
    Header.PolygonHeader.PointCountsOffset    = Header.NormalHeader.NormalsOffset + Header.NormalHeader.NormalCount * sizeof(VRLVertex);
    Header.PolygonHeader.FirstPointsOffset    = Header.PolygonHeader.PointCountsOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.NormalIndexesOffset  = Header.PolygonHeader.FirstPointsOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.DPlanesOffset      = Header.PolygonHeader.NormalIndexesOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.FlagsOffset      = Header.PolygonHeader.DPlanesOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.ColorsOffset      = Header.PolygonHeader.FlagsOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.MIPLevelsOffset    = Header.PolygonHeader.ColorsOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.TypesOffset      = Header.PolygonHeader.MIPLevelsOffset + Header.PolygonHeader.PolygonCount*4;
    Header.PolygonHeader.TexIndexesOffset    = Header.PolygonHeader.TypesOffset + Header.PolygonHeader.PolygonCount*4;

    NumVerts = Header.VertexHeader.VertexCount;
    NumPoly = Header.PolygonHeader.PolygonCount;
    NumPoints=Header.PointHeader.PointCount;
    NumNorms= Header.NormalHeader.NormalCount;
  }
  else
  {
    fHeader.DefsHeader.MinBox[0] = 0.0f; fHeader.DefsHeader.MinBox[1] = 0.0f; fHeader.DefsHeader.MinBox[2] = 0.0f;
    fHeader.DefsHeader.MaxBox[0] = 0.0f; fHeader.DefsHeader.MaxBox[1] = 0.0f; fHeader.DefsHeader.MaxBox[2] = 0.0f;
    fHeader.DefsHeader.Radius = 0.0f;
    fHeader.DefsHeader.LOD[0] = 0.0f;  fHeader.DefsHeader.LOD[1] = 0.0f;  fHeader.DefsHeader.LOD[2] = 0.0f;
    fHeader.DefsHeader.SwitchIn=0.0f;

    fHeader.DefsHeader.SwitchMid = 0x4cbebc20;
    fHeader.DefsHeader.SwitchOut = 0x4cbebc20;
    fHeader.DefsHeader.ZOffset  = 0.0f;
    fHeader.DefsHeader.nFlags  = 0;
    fHeader.VertexHeader.VertexCount      = pobj->VertTable->VertAmount;
    fHeader.VertexHeader.VertexsOffset      = 0xC8;
    fHeader.VertexHeader.TransformedVerexOffset  = fHeader.VertexHeader.VertexsOffset + fHeader.VertexHeader.VertexCount*sizeof(VRLVertex);
    fHeader.VertexHeader.GNormalsOffset      = fHeader.VertexHeader.TransformedVerexOffset + fHeader.VertexHeader.VertexCount*32;
    fHeader.VertexHeader.XNormalsOffset      = fHeader.VertexHeader.GNormalsOffset + fHeader.VertexHeader.VertexCount*16;
    fHeader.VertexHeader.VertexIntensitysOffset  = fHeader.VertexHeader.XNormalsOffset + fHeader.VertexHeader.VertexCount*16;
    fHeader.VertexHeader.VertexAIntensitysOffset  = fHeader.VertexHeader.VertexIntensitysOffset + fHeader.VertexHeader.VertexCount*4;
    fHeader.VertexHeader.VertexBIntensitysOffset  = fHeader.VertexHeader.VertexAIntensitysOffset + fHeader.VertexHeader.VertexCount*4;

    fHeader.PointHeader.PointCount        = pobj->FaceTable->FaceAmount*3;
    fHeader.PointHeader.PointsOffset      = fHeader.VertexHeader.VertexBIntensitysOffset + fHeader.VertexHeader.VertexCount*4;
    fHeader.PointHeader.TUPointsOffset      = fHeader.PointHeader.PointsOffset + fHeader.PointHeader.PointCount*4;
    fHeader.PointHeader.TVPointsOffset      = fHeader.PointHeader.TUPointsOffset + fHeader.PointHeader.PointCount*4;
    fHeader.PointHeader.TSPointsOffset      = fHeader.PointHeader.TVPointsOffset + fHeader.PointHeader.PointCount*4;
    fHeader.PointHeader.TTPointsOffset      = fHeader.PointHeader.TSPointsOffset + fHeader.PointHeader.PointCount*4;

    fHeader.NormalHeader.NormalCount      = pobj->FaceTable->FaceAmount;
    fHeader.NormalHeader.NormalsOffset      = fHeader.PointHeader.TTPointsOffset + fHeader.PointHeader.PointCount*4;

    fHeader.PolygonHeader.PolygonCount      = pobj->FaceTable->FaceAmount;
    fHeader.PolygonHeader.PointCountsOffset    = fHeader.NormalHeader.NormalsOffset + fHeader.NormalHeader.NormalCount * sizeof(VRLVertex);
    fHeader.PolygonHeader.FirstPointsOffset    = fHeader.PolygonHeader.PointCountsOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.NormalIndexesOffset  = fHeader.PolygonHeader.FirstPointsOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.DPlanesOffset      = fHeader.PolygonHeader.NormalIndexesOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.FlagsOffset      = fHeader.PolygonHeader.DPlanesOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.ColorsOffset      = fHeader.PolygonHeader.FlagsOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.MIPLevelsOffset    = fHeader.PolygonHeader.ColorsOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.TypesOffset      = fHeader.PolygonHeader.MIPLevelsOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.TexIndexesOffset    = fHeader.PolygonHeader.TypesOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.XMin          = fHeader.PolygonHeader.TexIndexesOffset + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.XMax          = fHeader.PolygonHeader.XMin + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.YMin          = fHeader.PolygonHeader.XMax + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.YMax          = fHeader.PolygonHeader.YMin + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.ZMin          = fHeader.PolygonHeader.YMax + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.PolygonHeader.ZMax          = fHeader.PolygonHeader.ZMin + fHeader.PolygonHeader.PolygonCount*4;

    NumVerts  = fHeader.VertexHeader.VertexCount;
    NumPoly    = fHeader.PolygonHeader.PolygonCount;
    NumPoints  = fHeader.PointHeader.PointCount;
    NumNorms  = fHeader.NormalHeader.NormalCount;

    PointsTS = new float[fHeader.PointHeader.PointCount*2];
    PolyZeros= new long[fHeader.PolygonHeader.PolygonCount];

  }

  ///////////////////////Calculate texture counts!!!,
  for (i = 0; i < d3d->Materials.MaterialsAmount; i++)
    MaterialMap[i] = -1;
  //resetted;
  j = 0;
  for (i = 0; i < pobj->FaceTable->FaceAmount; i++)
    if ((MaterialMap[pobj->FaceTable->Table[i].Material] == -1) &&
      (d3d->Materials.Materials[pobj->FaceTable->Table[i].Material].MatParams.PrimTexture >= 0))
    {//textured material first occure
      MaterialMap[pobj->FaceTable->Table[i].Material] = j;
      j++;
    }//textured material first occure
  NumTex    = j;//amount of textures
  Header.TextureMapHeader.TextureMapCount = NumTex;
  fHeader.TextureMapHeader.TextureMapCount = NumTex;

  
  if (j > 0)
  {// there are several textured materials
    TextureNames = new TextureName[j];
    for (i = 0; i < d3d->Materials.MaterialsAmount; i++)
    {//getting texture filenames
      if (MaterialMap[i] >= 0)
      {//textured material
        str = d3d->Materials.Materials[i].MaterialName;
        index = 0;
        index = str.ReverseFind(' ');
        if (index >= 0) str = str.Right(str.GetLength()-index-1);
        index = str.ReverseFind('.');
        if (index >= 0) str = str.Left(index);
        for (index=0; index < 12; index++)
          TextureNames[MaterialMap[i]][index] = '\0';
        //resetted
        strcpy(TextureNames[MaterialMap[i]], str.GetBuffer(12));
      }
    }//getting texture filenames
  }// there are several textured materials

  if (type == 1)
  {
    Header.TextureMapHeader.TextureMapsOffset  = Header.PolygonHeader.TexIndexesOffset + Header.PolygonHeader.PolygonCount*4;
    Header.NextObjectOffset            = 0;
  }
  else
    fHeader.TextureMapHeader.TextureMapsOffset  = fHeader.PolygonHeader.ZMax + fHeader.PolygonHeader.PolygonCount*4;
    fHeader.NextObjectOffset          = 0;
  
  //////////////////
  ////////////////// Header calculated
  //////////////////
  if (type == 1)  OutFile.Write(&Header, sizeof(VRLHeader));//writing header
  else      OutFile.Write(&fHeader, sizeof(fVRLHeader));//writing header
  
  verts = new VRLVertex[NumVerts];
  norms = new VRLVertex[NumNorms];
  VertZERObuf  = new long[NumVerts];
  VertIntens  = new long[NumVerts];
  if (type == 1)
  {
    FirstNorms  = new long[NumVerts];
    NormCounts  = new long[NumVerts];
    NormIndexes  = new long[NumVerts];
  }
  VPBlock.VertIndexes  = new long[NumPoints];
  VPBlock.Us      = new float[NumPoints];
  VPBlock.Vs      = new float[NumPoints];

  PolyCounts  = new long[NumPoly];
  FirstPolys  = new long[NumPoly];
  NormPolys  = new long[NumPoly];
  FlagsPolys  = new long[NumPoly];
  ColorsPolys  = new long[NumPoly];
  MIPPolys  = new long[NumPoly];
  TypesPolys  = new long[NumPoly];
  TexIndexes  = new long[NumPoly];
  DPlanePolys  = new float[NumPoly];

  for (i = 0 ; i < NumNorms; i++)
  {
    norms[i].X = 0.0f;
    norms[i].Y = 0.0f;
    norms[i].Z = 0.0f;
    norms[i].Updated  = 0;
  }

  for (i = 0 ; i < NumVerts; i++)
  {
    verts[i].X = pobj->VertTable->Table[i].X;
    verts[i].Y = pobj->VertTable->Table[i].Y;
    verts[i].Z = -pobj->VertTable->Table[i].Z;
    verts[i].Updated = 0;
    if (type == 1)
    {
      norms[i].X = -pobj->VertTable->Table[i].NormalX;
      norms[i].Y = -pobj->VertTable->Table[i].NormalY;
      norms[i].Z = pobj->VertTable->Table[i].NormalZ;
      norms[i].Updated  = 0;
      VertZERObuf[i]    = 0;
      VertIntens[i]    = 0x0f;
      FirstNorms[i]    = i;
      NormCounts[i]    = 1;//one normal per vertex
      NormIndexes[i]    = i;
    }
  }
  //calculating points
  for (i = 0; i < pobj->FaceTable->FaceAmount; i++)
  {
    VPBlock.VertIndexes[i*3]  = pobj->FaceTable->Table[i].I1;
    VPBlock.VertIndexes[i*3+1]  = pobj->FaceTable->Table[i].I3;
    VPBlock.VertIndexes[i*3+2]  = pobj->FaceTable->Table[i].I2;
    VPBlock.Us[i*3]        = pobj->FaceTable->Table[i].U1;
    VPBlock.Us[i*3+1]      = pobj->FaceTable->Table[i].U3;
    VPBlock.Us[i*3+2]      = pobj->FaceTable->Table[i].U2;
    VPBlock.Vs[i*3]        = pobj->FaceTable->Table[i].V1;
    VPBlock.Vs[i*3+1]      = pobj->FaceTable->Table[i].V3;
    VPBlock.Vs[i*3+2]      = pobj->FaceTable->Table[i].V2;
    
    if (type == 2)
    {
      PointsTS[i*6]        = 0.0f;
      PointsTS[i*6+1]        = 0.0f;
      PointsTS[i*6+2]        = 0.0f;
      PointsTS[i*6+3]        = 0.0f;
      PointsTS[i*6+4]        = 0.0f;
      PointsTS[i*6+5]        = 0.0f;
    }
  }
  
  //calculating polys
  for (i = 0; i < NumPoly; i++)
  {
    PolyCounts[i]  = 3;//3vertex poly
    FirstPolys[i]  = i*3;
    
    if (type == 1)
      NormPolys[i]  = Header.VertexHeader.VertexCount+i;
    else
      NormPolys[i] = i;

    Normal = GetNormal(
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I1].X,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I1].Y,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I1].Z,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I2].X,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I2].Y,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I2].Z,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I3].X,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I3].Y,
      pobj->VertTable->Table[pobj->FaceTable->Table[i].I3].Z);
    if (type == 1)
    {
      norms[Header.VertexHeader.VertexCount+i].X = Normal.x;
      norms[Header.VertexHeader.VertexCount+i].Y = Normal.y;
      norms[Header.VertexHeader.VertexCount+i].Z = Normal.z;
      norms[Header.VertexHeader.VertexCount+i].Updated = 0;
    }
    else
    {
      Normal.x = (  pobj->VertTable->Table[pobj->FaceTable->Table[i].I1].NormalX+
              pobj->VertTable->Table[pobj->FaceTable->Table[i].I2].NormalX+
              pobj->VertTable->Table[pobj->FaceTable->Table[i].I3].NormalX)/3.0f;
      Normal.y = (  pobj->VertTable->Table[pobj->FaceTable->Table[i].I1].NormalY+
              pobj->VertTable->Table[pobj->FaceTable->Table[i].I2].NormalY+
              pobj->VertTable->Table[pobj->FaceTable->Table[i].I3].NormalY)/3.0f;
      Normal.z = (  pobj->VertTable->Table[pobj->FaceTable->Table[i].I1].NormalZ+
              pobj->VertTable->Table[pobj->FaceTable->Table[i].I2].NormalZ+
              pobj->VertTable->Table[pobj->FaceTable->Table[i].I3].NormalZ)/3.0f;
      float len = VectLen(Normal.x, Normal.y, Normal.z);
      if (fabs(len) > MINFLOAT)
      {
        len = fdivide(1.0f, len);
        Normal.x*=len;
        Normal.y*=len;
        Normal.z*=len;
      }

      norms[i].X = -Normal.x;
      norms[i].Y = -Normal.y;
      norms[i].Z = Normal.z;
      norms[i].Updated = 0;
      
      PolyZeros[i]  = 0;
    }

    FlagsPolys[i]  = 0;
    MIPPolys[i]    = 0;
    
    if (MaterialMap[pobj->FaceTable->Table[i].Material] >=0)
    {//textured
      TexIndexes[i]  = MaterialMap[pobj->FaceTable->Table[i].Material];
      ColorsPolys[i]  = 0;
      if (type == 1)
      {//scgt types
        TypesPolys[i] = 0x34;//textured material
        if ((pobj->FaceTable->Table[i].nRenderFlags & SemiTransparent) == SemiTransparent)
          TypesPolys[i] = 0x0800;
        if (pobj->FaceTable->Table[i].HasFlag(Z3D_FLAG_MISC1))//use Alpha
          TypesPolys[i] |=0x02;
      }//scgt types
      else
      {
        if (pobj->FaceTable->Table[i].nRenderFlags & NoBlend)
          TypesPolys[i] = 0x34;
        else
          TypesPolys[i] = 0x9034;
        if (pobj->FaceTable->Table[i].nRenderFlags & SemiTransparent)//semitransparent
          TypesPolys[i] |= 0x0800;
      }
    }
    else
    {
      TexIndexes[i] = 0;
      TypesPolys[i] = 0x24;//NON textured material
      if ((type == 2) && ((pobj->FaceTable->Table[i].nRenderFlags & NoBlend)!=NoBlend))
        TypesPolys[i] |= 0x9000;//chrome
      ColorsPolys[i] = 
        (BYTE)(d3d->Materials.Materials[pobj->FaceTable->Table[i].Material].MatRec.dcvAmbient.r*255) +
        (BYTE)(d3d->Materials.Materials[pobj->FaceTable->Table[i].Material].MatRec.dcvAmbient.g*255)*256 +
        (BYTE)(d3d->Materials.Materials[pobj->FaceTable->Table[i].Material].MatRec.dcvAmbient.b*255)*256*256;
    }

    DPlanePolys[i]  = GetDPlane( verts[pobj->FaceTable->Table[i].I1],
                   verts[pobj->FaceTable->Table[i].I2],
                   verts[pobj->FaceTable->Table[i].I3]);
  }

  //vertices XYZ
  OutFile.Write(verts,    sizeof(VRLVertex)*NumVerts);

  //vertices Transformed
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);
  OutFile.Write(VertZERObuf,  4*NumVerts);

  if (type == 2)
  {
    OutFile.Write(VertZERObuf,  4*NumVerts);//G-norms
    OutFile.Write(VertZERObuf,  4*NumVerts);//G-norms
    OutFile.Write(VertZERObuf,  4*NumVerts);//G-norms
    OutFile.Write(VertZERObuf,  4*NumVerts);//G-norms

    OutFile.Write(VertZERObuf,  4*NumVerts);//X-norms
    OutFile.Write(VertZERObuf,  4*NumVerts);//X-norms
    OutFile.Write(VertZERObuf,  4*NumVerts);//X-norms
    OutFile.Write(VertZERObuf,  4*NumVerts);//X-norms
  }

  //vertices Intensity
  OutFile.Write(VertIntens,  4*NumVerts);
  //vertices AIntensity
  OutFile.Write(VertZERObuf,  4*NumVerts);
  //vertices BIntensity
  if (type == 1)
    OutFile.Write(VertZERObuf,  4*NumVerts);
  else
    OutFile.Write(VertIntens,  4*NumVerts);

  if (type == 1)
  {
    //vertices FirstGNorms
    OutFile.Write(FirstNorms,  4*NumVerts);
    //vertices GNormsCounts
    OutFile.Write(NormCounts,  4*NumVerts);
    //vertices NormIndexes
    OutFile.Write(NormIndexes,  4*NumVerts);
    //vertices writeDone
  }

  //writing points
  //points Indexes
  OutFile.Write(VPBlock.VertIndexes, 4*NumPoints);
  //points U
  OutFile.Write(VPBlock.Us,  4*NumPoints);
  //points V
  OutFile.Write(VPBlock.Vs,  4*NumPoints);
  if (type == 2)
    OutFile.Write(PointsTS,  8*NumPoints);
  //points writeDone
  
  //writing normals
  //normals XYZupdate
  OutFile.Write(norms,    sizeof(VRLVertex)*NumNorms);
  //normals writeDone

  //writing Polys
  //PointsCount
  OutFile.Write(PolyCounts,  4*NumPoly);
  //FirstPolys
  OutFile.Write(FirstPolys,  4*NumPoly);
  //NormalIndexes
  OutFile.Write(NormPolys,  4*NumPoly);
  //DPlanes
  OutFile.Write(DPlanePolys,  4*NumPoly);
  //Flags
  OutFile.Write(FlagsPolys,  4*NumPoly);
  //Colors
  OutFile.Write(ColorsPolys,  4*NumPoly);
  //MIP
  OutFile.Write(MIPPolys,    4*NumPoly);
  //Types
  OutFile.Write(TypesPolys,  4*NumPoly);
  //Textureindexes
  OutFile.Write(TexIndexes,  4*NumPoly);
  if (type == 2)
  {
    OutFile.Write(PolyZeros,  4*NumPoly);//XMin
    OutFile.Write(PolyZeros,  4*NumPoly);
    OutFile.Write(PolyZeros,  4*NumPoly);
    OutFile.Write(PolyZeros,  4*NumPoly);
    OutFile.Write(PolyZeros,  4*NumPoly);
    OutFile.Write(PolyZeros,  4*NumPoly);//ZMax
  }
  //Polygons writeDone

  //writing texturenames
  //texturenames
  if (TextureNames)
    OutFile.Write(TextureNames,  sizeof(TextureName)*NumTex);
  //normals writeDone

  delete[] verts;
  delete[] norms;
  delete[] VertIntens;
  delete[] VertZERObuf;
  if (type == 1)
  {
    delete[] FirstNorms;
    delete[] NormCounts;
    delete[] NormIndexes;
  }
  delete[] VPBlock.VertIndexes;
  delete[] VPBlock.Us;
  delete[] VPBlock.Vs;

  delete[] PolyCounts;
  delete[] FirstPolys;
  delete[] NormPolys;
  delete[] DPlanePolys;
  delete[] FlagsPolys;
  delete[] ColorsPolys;
  delete[] MIPPolys;
  delete[] TypesPolys;
  delete[] TexIndexes;
  
  if (TextureNames) delete[] TextureNames;
  if (type == 2)
  {
    delete[] PolyZeros;
    delete[] PointsTS;
  }

  return Z3D_PLUGRESULT_SELECTIONCHANGED;
}

/**********************************************************************/
// This filter supports multiple files export
/**********************************************************************/
DWORD CALLBACK MultipleExportInfo(DWORD num)
{
  if (num == 0) //SCGT
    return ZMULTIPLE_FILES_EXPORT_NEED_FILES | ZMULTIPLE_FILES_EXPORT_NEED_ONEOBJECT | ZMULTIPLE_FILES_EXPORT_SELECTED;

  if (num == 1) //F1
    return ZMULTIPLE_FILES_EXPORT_NEED_FILES | ZMULTIPLE_FILES_EXPORT_NEED_ONEOBJECT | ZMULTIPLE_FILES_EXPORT_SELECTED;

  return 0;
}

char* CALLBACK MultipleExportTitle(DWORD num)
{
  if (num == 0) //SCGT
    return "Sports Car GT VRL";
  if (num == 1) //F1
    return "Forumla One 2000 VRL";
  return NULL;
}

char* CALLBACK MultipleExportHint(DWORD num)
{
  return "Select object(s) for exporting.";
}


DWORD CALLBACK MultipleExportSportsCarGTVRL(CFile* pFile, tObject* pObject, CDirect3D* pD3D)
{
  return MultipleExport(*(CFile* &)pFile, pObject, pD3D, 1);
}

DWORD CALLBACK MultipleExportForumlaOne2000VRL(CFile* pFile, tObject* pObject, CDirect3D* pD3D)
{
  return MultipleExport(*(CFile* &)pFile, pObject, pD3D, 2);
}